<?php

namespace MPHB\Crons;

class CronManager {

	const INTERVAL_PENDING_USER_APPROVAL = 'mphb_pending_user_approval';
	const INTERVAL_PENDING_PAYMENT       = 'mphb_pending_payment';
	const INTERVAL_AUTODELETE_SYNC_LOGS  = 'mphb_ical_auto_delete';

	const INTERVAL_QUARTER_AN_HOUR = 'mphb_15m';
	const INTERVAL_HALF_AN_HOUR    = 'mphb_30m';
	/** @since 5.0.0 */
	const INTERVAL_WEEKLY          = 'mphb_weekly';

	// Default WordPress intervals
	const INTERVAL_DAILY       = 'daily';
	const INTERVAL_TWICE_DAILY = 'twicedaily';
	const INTERVAL_HOURLY      = 'hourly';

	/**
	 * @var Cron[]
	 */
	private $crons = array();

	public function __construct() {

		add_filter( 'cron_schedules', array( $this, 'createCronIntervals' ) );

		$this->initCrons();

		// schedule all necessary crons
		// MPHB\Libraries\WP_SessionManager starts its own cron

		$this->getCron( 'check_license_status' )->schedule();

	}

	/**
	 * @since 3.6.1 added new cron - DeleteOldSyncLogsCron.
	 * @since 5.0.0 added new cron - CheckLicenseStatusCron.
	 */
	public function initCrons() {

		$crons = array(
			new AbandonBookingPendingUserCron(
				'abandon_booking_pending_user',
				self::INTERVAL_PENDING_USER_APPROVAL
			),
			new AbandonBookingPendingPaymentCron(
				'abandon_booking_pending_payment',
				self::INTERVAL_PENDING_PAYMENT
			),
			new AbandonPaymentPendingCron(
				'abandon_payment_pending',
				self::INTERVAL_PENDING_PAYMENT
			),
			new CheckLicenseStatusCron(
				'check_license_status',
				self::INTERVAL_WEEKLY
			),
		);

		foreach ( $crons as $cron ) {
			$this->addCron( $cron );
		}
	}

	public function addCron( AbstractCron $cron ): void {
		$this->crons[ $cron->getId() ] = $cron;
	}

	public function getCron( string $id ): ?AbstractCron {
		return isset( $this->crons[ $id ] ) ? $this->crons[ $id ] : null;
	}

	/**
	 * @param array $schedules
	 * @return array
	 *
	 * @since 3.6.1 added new interval - "Interval for automatic cleaning of synchronization logs".
	 * @since 3.6.1 added new interval - "Quarter an Hour".
	 * @since 3.6.1 added new interval - "Half an Hour".
	 */
	public function createCronIntervals( $schedules ) {

		$schedules[ self::INTERVAL_QUARTER_AN_HOUR ] = array(
			'interval' => 15 * MINUTE_IN_SECONDS,
			'display'  => 'Quarter an Hour',
		);

		$schedules[ self::INTERVAL_HALF_AN_HOUR ] = array(
			'interval' => 30 * MINUTE_IN_SECONDS,
			'display'  => 'Half an Hour',
		);

		$schedules[ self::INTERVAL_PENDING_USER_APPROVAL ] = array(
			'interval' => MPHB()->settings()->main()->getUserApprovalTime() * MINUTE_IN_SECONDS,
			'display'  => 'User Approval Time setted in Hotel Booking Settings',
		);

		$schedules[ self::INTERVAL_PENDING_PAYMENT ] = array(
			'interval' => MPHB()->settings()->payment()->getPendingTime() * MINUTE_IN_SECONDS,
			'display'  => 'Pending Payment Time set in Hotel Booking Settings',
		);

		$schedules[ self::INTERVAL_AUTODELETE_SYNC_LOGS ] = array(
			'interval' => 6 * HOUR_IN_SECONDS,
			'display'  => 'Interval for automatic cleaning of synchronization logs.',
		);

		$schedules[ self::INTERVAL_WEEKLY ] = array(
			'interval' => WEEK_IN_SECONDS,
			'display'  => 'Once a week',
		);

		return $schedules;
	}


	public function do_on_plugin_deactivation() {

		if ( ! empty( $this->crons ) ) {

			foreach ( $this->crons as $cron ) {

					
				$cron->unschedule();
			}
		}

		wp_clear_scheduled_hook( 'mphb_wp_session_garbage_collection' );
	}
}
